/*****************************************************************************
 *
 *  fs_ltt.c - Defines methods of .ltt internal structure.
 *
 *  Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved.
 *
 *  Confidential information of Monotype Imaging Inc.
 *
 ****************************************************************************/


#include "fs_object.h"
#include "fs_function.h"
#ifdef FS_LINKED_FONTS
#include "fs_ltt.h"


#define DEFAULT_BOLD_PERCENT   ( 6 << 16)   /* 6% of em box */
#define DEFAULT_ITALIC_ANGLE   (15 << 16)   /* 15 degrees for pseudo-italic */
#ifdef STROKE_PERCENT_PER_COMPONENT_IS_DESIRED
#define DEFAULT_STROKE_PERCENT ( 5 << 16)   /* 5% of em box */
#endif

#ifndef FALSE
#define FALSE 0
#endif

FS_ERROR
FsLttComponent_init(FsLttComponent *lttc, FsLtt *ltt)
{
    lttc->fontName = NULL;
    lttc->fileName = NULL;

    lttc->bmp.range1 = 0;
    lttc->bmp.range2 = 0;

    lttc->numGlyphs = 0;
    lttc->numIcons = 0;

    lttc->ttcIndex = 0;

    lttc->targetCId = NULL;
    lttc->targetDir = NULL;
    lttc->targetPtr = 0;
    lttc->targetOff = 0;
    lttc->targetLen = 0;

    lttc->fontOffset = 0;
    lttc->fontSize = 0;

    lttc->bold = FALSE;
    lttc->boldPercent = DEFAULT_BOLD_PERCENT;

    lttc->italic = FALSE;
    lttc->italicAngle = DEFAULT_ITALIC_ANGLE;

#ifdef STROKE_PERCENT_PER_COMPONENT_IS_DESIRED
    lttc->strokePercent = DEFAULT_STROKE_PERCENT;
#endif

    lttc->smp.range1 = 0;
    lttc->smp.range2 = 0;
    lttc->sip.range1 = 0;
    lttc->sip.range2 = 0;
    lttc->ssp.range1 = 0;
    lttc->ssp.range2 = 0;

    lttc->adjustmentsOffset = 0;
    lttc->numAdjustments = 0;
    lttc->adjustments = NULL;
    lttc->adjustmentsCapacity = 0;

    lttc->group = 0;

    lttc->winPath = NULL;

    lttc->ltt = ltt;

    return SUCCESS;
}

FS_ERROR
FsLttComponent_done(_DS_ FsLttComponent *lttc)
{
    if (lttc->fontName) FSS_free(_PS_ lttc->fontName);

    if (lttc->fileName) FSS_free(_PS_ lttc->fileName);

    if (lttc->targetDir) FSS_free(_PS_ lttc->targetDir);

    if (lttc->targetCId) FSS_free(_PS_ lttc->targetCId);

    if (lttc->adjustments) FSS_free(_PS_ lttc->adjustments);

    if (lttc->winPath) FSS_free(_PS_ lttc->winPath);

    return SUCCESS;
}

FS_BOOLEAN
FsLttComponent_isBold(FsLttComponent *lttc, FS_FIXED *boldPercent)
{
    *boldPercent = lttc->boldPercent;
    return lttc->bold;
}

FS_BOOLEAN
FsLttComponent_isItalic(FsLttComponent *lttc, FS_FIXED *italicAngle)
{
    *italicAngle = lttc->italicAngle;
    return lttc->italic;
}

FS_USHORT
FsLttComponent_numAdjustments(FsLttComponent *lttc)
{
    return lttc->numAdjustments;
}

FS_CONST FsPPemAdjust *
FsLttComponent_getAdjustment(FsLttComponent *lttc, FS_USHORT n)
{
    /* loading adjustments is an option when reading/loading */

    if (lttc->adjustments &&
        (lttc->numAdjustments > 0) && 
        (n < lttc->numAdjustments))
    {
        return lttc->adjustments + n;
    }
    else
    {
        return NULL;
    }
}

FS_ERROR
FsLttComponent_getAdjustmentAndRange(FsLttComponent *lttc,
                                     FsPPemAdjust *adj, FS_USHORT *maxppem)
{
    /* loading adjustments is an option when reading/loading */

    if (lttc->adjustments || !lttc->numAdjustments)
    {
        FS_USHORT target;

        target = adj->ppem;

        adj->ppem = 0;
        adj->scale = 0;
        adj->shift = 0;
        *maxppem = 0xFFFF;

        if (lttc->numAdjustments)
        {
            FS_USHORT i;
            for (i = 0; i < lttc->numAdjustments; i++)
            {
                FsPPemAdjust *p = lttc->adjustments + i;

                if (p->ppem > target)
                {
                    *maxppem = p->ppem - 1;
                    break;
                }
                else
                {
                    *adj = *p;
                    *maxppem = 0xFFFF;
                }
            }
        }
        return SUCCESS;
    }
    else
    {
        return ERR_INVALID_ADDRESS;
    }
}

static FS_VOID
FsLttOtTable_init(FsLttOtTable *t)
{
    t->offset = 0;
    t->size = 0;
}

FS_ERROR
FsLtt_init(_DS_ FsLtt *ltt)
{
    ltt->version = 0;

    ltt->numLegal = 0;

    ltt->legal = NULL;

    ltt->name = NULL;
    ltt->subfamilyName = NULL;

    ltt->fileversion = NULL;

    ltt->numComponents = 0;
    ltt->mtrxComponent = 0xFFFF;
    ltt->components = NULL;

    ltt->ttf_name.offset = 0;
    ltt->ttf_name.size   = 0;
    ltt->maxp_off = 0;
    ltt->os_2_off = 0;
    ltt->head_off = 0;
    ltt->hhea_off = 0;
    ltt->vhea_off = 0;
    ltt->post_off = 0;

    FsLttOtTable_init(&ltt->cmap);

    FsLttOtTable_init(&ltt->gdef);
    FsLttOtTable_init(&ltt->gsub);
    FsLttOtTable_init(&ltt->gpos);

    ltt->groupTypesOffset = 0;
    ltt->numGroupTypes = 0;
    ltt->groupTypes = NULL;
    ltt->groupTypesCapacity = 0;

    /**/
    ltt->capacityComponents = 0;

    ltt->fsState = FS_state_ptr;

    return SUCCESS;
} /*lint !e429 Suppress 'Custodial pointer 'ltt' has not been freed or returned' */

FsLtt *
FsLtt_new(_DS0_)
{
    FsLtt *ltt;

    ltt = FSS_malloc(_PS_ sizeof(FsLtt));
    if (!ltt) return NULL;

    FsLtt_init(_PS_ ltt);

    return ltt;
}

FS_VOID
FsLtt_done(FsLtt *ltt)
{
    FS_USHORT i;
    LTT_STATE_PTR;

    for (i = 0; i < ltt->numLegal; i++)
    {
        if (ltt->legal[i]) FSS_free(_PS_ ltt->legal[i]);
    }
    if (ltt->legal) FSS_free(_PS_ ltt->legal);

    if (ltt->name) FSS_free(_PS_ ltt->name);
    if (ltt->subfamilyName) FSS_free(_PS_ ltt->subfamilyName);

    if (ltt->fileversion) FSS_free(_PS_ ltt->fileversion);

    for (i = 0; i < ltt->numComponents; i++)
    {
        FsLttComponent_done(_PS_ ltt->components + i);
    }
    if (ltt->components) FSS_free(_PS_ ltt->components);

    if (ltt->groupTypes) FSS_free(_PS_ ltt->groupTypes);
}

FS_VOID
FsLtt_delete(FsLtt *ltt)
{
    LTT_STATE_PTR;

    FsLtt_done(ltt);

    FSS_free(FS_state_ptr, ltt);
}

FS_CONST FILECHAR *
FsLtt_getName(FsLtt *ltt)
{
    return ltt->name;
}

FS_CONST FILECHAR *
FsLtt_getSubfamilyName(FsLtt *ltt)
{
    return ltt->subfamilyName ? ltt->subfamilyName : "Regular";
}

FS_CONST FILECHAR *
FsLtt_getFileVersion(FsLtt *ltt)
{
    return ltt->fileversion;
}

FS_USHORT
FsLtt_numComponents(FsLtt *ltt)
{
    return ltt->numComponents;
}

FsLttComponent *
FsLtt_component(FsLtt *ltt, FS_USHORT ind)
{
    if ((ltt->numComponents > 0) && (ind < ltt->numComponents))
    {
        return ltt->components + ind;
    }
    else
    {
        return NULL;
    }
}

FS_USHORT
FsLtt_getMetric(FsLtt *ltt)
{
    return ltt->mtrxComponent;
}

FS_USHORT
FsLtt_getGroupType(FsLtt *ltt, FS_USHORT group)
{
    FS_USHORT i;

    if (group == 0)
    {
        return FsLTT_GROUPTYPE_0;
    }
    for (i = 0; i < ltt->numGroupTypes; i++)
    {
        FsGroupType *p = ltt->groupTypes + i;

        if (p->group == group)
        {
            return p->type;
        }
        else if (p->group > group)
        {
            break;
        }
    }
    return FsLTT_GROUPTYPE_0;
}

#endif /* FS_LINKED_FONTS */
